<?php

/**
 * Unschedule a previously scheduled cron job using job key.
 *
 * @param int $timestamp timestamp for when to run the event.
 * @param string $hook action hook, the execution of which will be unscheduled.
 * @param string $key key for arguments to identify the event.
 */
function gd_unschedule_event($timestamp, $hook, $key) {
	$crons = _get_cron_array();
	unset( $crons[$timestamp][$hook][$key] );
	if ( empty($crons[$timestamp][$hook]) )
		unset( $crons[$timestamp][$hook] );
	if ( empty($crons[$timestamp]) )
		unset( $crons[$timestamp] );
	_set_cron_array( $crons );
}

/**
 * Get shortlink for any post or page.
 *
 * @global GDPressTools $gdpt main plugin object instance
 * @global object $post wordpress post object
 * @param int $post_id id of post or page to get url
 * @return string shortened url
 */
function gd_get_shortlink($post_id = 0) {
    global $gdpt;
    if ($post_id == 0) {
        global $post;
        $post_id = $post->ID;
    }
    return $gdpt->get_shortlink($post_id);
}

/**
 * Return WordPress option without using cache.
 */
function gd_get_option_force($setting) {
    global $wpdb;
    $row = $wpdb->get_row("SELECT option_value FROM $wpdb->options WHERE option_name = '$setting' LIMIT 1");
    $value = is_object($row) ? $row->option_value : "";
    return apply_filters('option_'.$setting, maybe_unserialize($value));
}

/**
 * Return object with page loading stats to a point where the function is called.
 */
function gd_page_stats() {
    return new gdptStats();
}

/**
 * Message to dispay to disabled RSS feeds.
 */
function gd_disable_feed() {
    $feed = __("RSS feed is not available. Instead, please visit our website: ", "gd-press-tools");
    $feed.= '<a href="'.get_bloginfo('url').'">'.get_option('blogname').'</a>';
    wp_die($feed);
}

/**
 * Change the status of post revisions saving
 *
 * @param int $status Status to set (-1, 0, >0)
 */
function gd_set_revisions($status) {
    define('WP_POST_REVISIONS', $status);
}

/**
 * Resets sidebars widgets
 */
function gd_reset_widgets() {
    update_option('sidebars_widgets', $null);
}

/**
 * Optimizes WordPress database tables.
 *
 * @global WPDB $wpdb wordpress database object
 */
function gd_optimize_db() {
    global $wpdb;
    $tables = gd_get_wordpress_tables();
    foreach ($tables as $t) {
        $wpdb->query("OPTIMIZE TABLE `".$t."`");
        $wpdb->query("REPAIR TABLE `".$t."`");
    }
}

/**
 * Clear RSS Feeds cache from the database.
 *
 * @global WPDB $wpdb wordpress database object
 * @global string $table_prefix prefix used for database tables
 */
function gd_clear_rss_cache() {
    global $wpdb, $table_prefix;
    $sql = sprintf("DELETE FROM %soptions WHERE option_name LIKE '%s' and LENGTH(option_name) IN (36, 39)",
        $table_prefix, "rss\_%");
    wp_gdpt_log_sql("CLEAR_RSS_CACHE", $sql);
    $wpdb->query($sql);
}

/**
 * Gets the list of all WordPress tables.
 *
 * @global WPDB $wpdb wordpress database object
 * @global string $table_prefix prefix used for database tables
 * @return array list of wordpress tables
 */
function gd_get_wordpress_tables() {
    global $wpdb, $table_prefix;
    $sql = "SHOW TABLES LIKE '".str_replace("_", "\_", $table_prefix)."%'";
    $tables = $wpdb->get_results($sql, ARRAY_N);
    $result = array();
    foreach ($tables as $t) $result[] = $t[0];
    if ($wpdb->usermeta != $table_prefix."usermeta") $result[] = $wpdb->usermeta;
    if ($wpdb->users != $table_prefix."users") $result[] = $wpdb->users;
    return $result;
}

/**
 * Gets the list of all tables in active database.
 *
 * @global WPDB $wpdb wordpress database object
 * @global string $table_prefix prefix used for database tables
 * @return array list of wordpress tables
 */
function gd_get_database_tables() {
    global $wpdb, $table_prefix;
    $sql = "SHOW TABLES";
    $tables = $wpdb->get_results($sql, ARRAY_N);
    $result = array();
    foreach ($tables as $t) $result[] = $t[0];
    return $result;
}

/**
 * Gets the array with columns of the database table.
 *
 * @global WPDB $wpdb wordpress database object
 * @global string $table_prefix prefix used for database tables
 * @return array table columns
 */
function gd_get_database_table_columns($table) {
    global $wpdb, $table_prefix;
    $sql = "SHOW COLUMNS FROM ".$table;
    return $wpdb->get_results($sql);
}

/**
 * Drop database table.
 *
 * @global WPDB $wpdb wordpress database object
 * @global string $table_prefix prefix used for database tables
 * @return array table columns
 */
function gd_db_table_drop($table) {
    global $wpdb, $table_prefix;
    $sql = "DROP TABLE ".$table;
    wp_gdpt_log_sql("DROP_TABLE", $sql);
    $wpdb->query($sql);
}

/**
 * Empty database table.
 *
 * @global WPDB $wpdb wordpress database object
 * @global string $table_prefix prefix used for database tables
 * @return array table columns
 */
function gd_db_table_empty($table) {
    global $wpdb, $table_prefix;
    $sql = "TRUNCATE ".$table;
    wp_gdpt_log_sql("TRUNCATE_TABLE", $sql);
    $wpdb->query($sql);
}

/**
 * Get total number of revisions for all posts.
 *
 * @global WPDB $wpdb wordpress database object
 * @global string $table_prefix prefix used for database tables
 * @return int number of revisions
 */
function gd_count_revisions_total() {
    global $wpdb, $table_prefix;

    $sql = sprintf("select count(*) as revisions from %sposts p inner join %sposts r on p.post_parent = r.ID where p.post_type = 'revision' and r.post_status = 'publish'", $table_prefix, $table_prefix);
    return $wpdb->get_var($sql);
}

/**
 * Get the number of spam comments for all posts.
 *
 * @return int number of spam comments
 */
function gd_count_spam_total() {
    $cmm = wp_count_comments();
    return $cmm->spam;
}

/**
 * Get revisions count for the post.
 *
 * @global WPDB $wpdb wordpress database object
 * @global string $table_prefix prefix used for database tables
 * @param int $post_id post id to get revisions from
 * @return int number of revisions
 */
function gd_count_revisions($post_id) {
    global $wpdb, $table_prefix;

    $sql = sprintf("select count(*) as revisions from %sposts where post_type = 'revision' and post_parent = %s", $table_prefix, $post_id);
    return $wpdb->get_var($sql);
}

/**
 * Get views count for the post.
 *
 * @global WPDB $wpdb wordpress database object
 * @global string $table_prefix prefix used for database tables
 * @param int $post_id post id to get revisions from
 * @return object views results
 */
function gd_count_views($post_id) {
    global $wpdb, $table_prefix;

    $sql = sprintf("select p.ID as post_id, sum(v.usr_views) as usr_views, sum(v.vst_views) as vst_views, sum(v.usr_views + v.vst_views) as tot_views from %sposts p left join %sgdpt_posts_views v on v.post_id = p.ID where p.ID = %s group by p.ID",
        $table_prefix, $table_prefix, $post_id);
    return $wpdb->get_row($sql);
}

/**
 * Deletes all revisions for all published posts.
 *
 * @global WPDB $wpdb wordpress database object
 * @global string $table_prefix prefix used for database tables
 * @return int number of deleted revisions
 */
function gd_delete_all_revisions() {
    global $wpdb, $table_prefix;

    $sql = sprintf("delete %s, %s, %s from %sposts p inner join %sposts r on p.post_parent = r.ID left join %sterm_relationships t on t.object_id = p.ID left join %spostmeta m on m.post_id = p.ID where p.post_type = 'revision' and r.post_status = 'publish'",
        gdFunctionsGDPT::mysql_pre_4_1() ? sprintf("%sposts", $table_prefix) : "p",
        gdFunctionsGDPT::mysql_pre_4_1() ? sprintf("%sterm_relationships", $table_prefix) : "t",
        gdFunctionsGDPT::mysql_pre_4_1() ? sprintf("%spostmeta", $table_prefix) : "m",
        $table_prefix, $table_prefix, $table_prefix, $table_prefix);
    $wpdb->query($sql);
    wp_gdpt_log_sql("DELETE_REVISIONS_ALL", $sql);
    return $wpdb->rows_affected;
}

/**
 * Deletes revisions for selected post.
 *
 * @global WPDB $wpdb wordpress database object
 * @global string $table_prefix prefix used for database tables
 * @param int $post_id post id to delete revisions from
 * @return int number of deleted revisions
 */
function gd_delete_revisions($post_id) {
    global $wpdb, $table_prefix;

    $sql = sprintf("delete %s, %s, %s from %sposts p left join %sterm_relationships t on t.object_id = p.ID left join %spostmeta m on m.post_id = p.ID where p.post_type = 'revision' and p.post_parent = %s",
        gdFunctionsGDPT::mysql_pre_4_1() ? sprintf("%sposts", $table_prefix) : "p",
        gdFunctionsGDPT::mysql_pre_4_1() ? sprintf("%sterm_relationships", $table_prefix) : "t",
        gdFunctionsGDPT::mysql_pre_4_1() ? sprintf("%spostmeta", $table_prefix) : "m",
        $table_prefix, $table_prefix, $table_prefix, $post_id);
    $wpdb->query($sql);
    wp_gdpt_log_sql("DELETE_REVISIONS_FOR_POST", $sql);
    return $wpdb->rows_affected;
}

/**
 * Creates index.php for protecting wp-content folders.
 */
function gd_create_protection_files() {
    $wp["plugins"] = trailingslashit(dirname(PRESSTOOLS_PATH));
    $wp["content"] = trailingslashit(dirname($wp["plugins"]));
    $wp["themes"] = $wp["content"]."themes/";

    $index = file_get_contents(PRESSTOOLS_PATH."files/index.php");
    $index = str_replace('%URL%', get_bloginfo("url"), $index);

    foreach ($wp as $folder) {
        $path = $folder."index.php";
        if (!file_exists($path) && is_writable($folder)) {
            $f = fopen($path, "a+");
            fwrite ($f, "$index");
            fclose($f);
        }
    }
}

/**
 * Creates index.php for protecting single folder.
 */
function gd_create_protection_file($folder) {
    $index = file_get_contents(PRESSTOOLS_PATH."files/index.php");
    $index = str_replace('%URL%', get_bloginfo("url"), $index);
    $path = $folder."index.php";
    if (!file_exists($path) && is_writable($folder)) {
        $f = fopen($path, "a+");
        fwrite ($f, "$index");
        fclose($f);
    }
}

?>